import Foundation
import UIKit
import SafariServices
import DCloudUTSFoundation

enum PaymentStatus: Int {
    case unpaid = 0
    case succeeded = 1
    case failed = 2
}

@objcMembers class JPayAPM: NSObject, SFSafariViewControllerDelegate {
    
    private var payStatus: PaymentStatus = .unpaid
    private var isCallbackCalled: Bool = false
    
    /// Execute payment request
    /// - Parameters:
    ///   - payMethod: Payment method identifier
    ///   - url: Payment URL
    ///   - callback: Callback function
    public func exec(_ payMethod: String, _ url: String, _ callback: @escaping (UTSJSONObject) -> Void) {
        console.log("Execute payment request: payMethod=\(payMethod), url=\(url)")
        // 重置状态
        self.payStatus = .unpaid
        self.isCallbackCalled = false
        
        guard let paymentUrl = URL(string: url) else {
            let errorMsg = "Invalid URL format: \(url)"
            console.log(errorMsg)
            // 创建UTSJSONObject并调用回调
            let result = UTSJSONObject(dictionary: [
                "status": NSInteger(PaymentStatus.failed.rawValue),
                "message": errorMsg,
                "orderId": ""
            ])
            callback(result)
            return
        }
        
        DispatchQueue.main.async {
            self.showPaymentBrowser(with: paymentUrl)
        }
    }
    
    /// Display payment browser
    /// - Parameter url: Payment URL
    private func showPaymentBrowser(with url: URL) {
        let configuration = SFSafariViewController.Configuration()
        configuration.barCollapsingEnabled = true
        
        // 创建SafariViewController并持有强引用
        let safariVC = SFSafariViewController(url: url, configuration: configuration)
        safariVC.dismissButtonStyle = .done
        safariVC.delegate = self
        
        let currentVC = UTSiOS.getCurrentViewController()
        currentVC.present(safariVC, animated: true) {
            console.log("SafariViewController presented, URL: \(url.absoluteString)")
        }
    }
}